#ifndef IAnalytics_h
#define IAnalytics_h
//
//  Copyright (c) 2020 Microsoft. All rights reserved.
//

#import "EIAction.h"
#import "EIEvent.h"
#import "EIView.h"
#import "EIUser.h"

#import <Foundation/Foundation.h>

/*!
 @brief EIIAnalytics
*/
@protocol EIIAnalytics <NSObject>

#pragma mark Telemetry tracking methods

/*!
 @brief Tracks a custom event.
 @param event A EIEvent object
 */
-(void)trackEvent:(nonnull EIEvent *)event;

/*!
@brief Tracks a EIAction event.
@param action A EIAction object
*/
-(void)trackAction:(nonnull EIAction *)action;

/*!
@brief Tracks a EIView event.
@param view A EIView object
*/
-(void)trackView:(nonnull EIView *)view;

#pragma mark Set EIUser method

/*!
@brief Adds (or overrides) the user properties of the event for the telemetry tracking system.
User information applies to events generated by this Analytics instance.
@param user A EIUser object
*/
-(void)setUser:(nonnull EIUser *)user;

#pragma mark Set custom property methods 

/*!
@brief Adds (or overrides) a property of the event for the telemetry tracking system.
Property information applies to events generated by this Analytics instance, unless it is overwritten on a particular event property.
@param value Contains the desired property value. The supported value types are NSString, NSDate and NSUUID.
@param key A string that contains the name of the property.
*/
-(void)setValue:(nullable id)value forKey:(nonnull NSString *)key;

/*!
 @brief Adds (or overrides) a property of the event for the telemetry tracking system.
 Property information applies to events generated by this Analytics instance, unless it is overwritten on a particular event property.
 @param key A string that contains the name of the property.
 @param value A boolean that contains the property value.
 */
-(void)setBoolValue:(BOOL)value forKey:(nonnull NSString *)key;

/*!
 @brief Adds (or overrides) a property of the event for the telemetry tracking system.
 Property information applies to events generated by this Analytics instance, unless it is overwritten on a particular event property.
 @param key A string that contains the name of the property.
 @param value A double that contains the property value.
 */
-(void)setDoubleValue:(double)value forKey:(nonnull NSString *)key;

/*!
 @brief Adds (or overrides) a property of the event for the telemetry tracking system.
 Property information applies to events generated by this Analytics instance, unless it is overwritten on a particular event property.
 @param key A string that contains the name of the property.
 @param value An int64_t that contains the property value.
 */
-(void)setLongValue:(int64_t)value forKey:(nonnull NSString *)key;

@end

#endif // IAnalytics_h

